<?php

namespace App\Http\Controllers\Setting;

use App\Http\Controllers\Controller;
use App\Models\Branches;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;

class UserController extends Controller
{
    public function index()
    {
        $user = Auth::user();

        $users = User::where('clinic_id', $user->clinic_id)
            ->orderBy('id', 'desc')
            ->paginate(20);

        return view('admin.users.index', compact('users'));
    }

    public function create()
    {
        $current = Auth::user();
        $branches = Branches::where('clinic_id', $current->clinic_id)->get();

        return view('admin.users.create', compact('branches'));
    }

    public function store(Request $request)
    {
        $current = Auth::user();

        $data = $request->validate([
            'name'      => 'required|string|max:100',
            'email'     => 'required|email|unique:users,email',
            'phone'     => 'nullable|string|max:20',
            'password'  => 'required|min:6',
            'role'      => 'required|in:owner,doctor,receptionist,accountant,nurse,admin',
            'branch_id' => 'nullable|exists:branches,id',
        ]);

        User::create([
            'name'      => $data['name'],
            'email'     => $data['email'],
            'password'  => Hash::make($data['password']),
            'phone'     => $data['phone'],
            'role'      => $data['role'],
            'clinic_id' => $current->clinic_id,
            'branch_id' => $data['branch_id'],
            'is_active' => true,
        ]);

        return redirect()->route('settings.users.index')->with('success', 'تم إضافة المستخدم بنجاح.');
    }

    public function edit($id)
    {
        $current = Auth::user();

        $user = User::where('clinic_id', $current->clinic_id)->findOrFail($id);
        $branches = Branches::where('clinic_id', $current->clinic_id)->get();


        return view('admin.users.edit', compact('user', 'branches'));
    }

    public function update(Request $request, $id)
    {
        $current = Auth::user();

        $user = User::where('clinic_id', $current->clinic_id)->findOrFail($id);

        $data = $request->validate([
            'name'      => 'required|string|max:100',
            'email'     => 'required|email|unique:users,email,' . $user->id,
            'phone'     => 'nullable|string|max:20',
            'password'  => 'nullable|min:6',
            'role'      => 'required|in:owner,doctor,receptionist,accountant,nurse,admin',
            'branch_id' => 'nullable|exists:branches,id',

        ]);

        $user->name     = $data['name'];
        $user->email    = $data['email'];
        $user->phone    = $data['phone'];
        $user->role     = $data['role'];
        $user->branch_id = $data['branch_id'];


        if (!empty($data['password'])) {
            $user->password = Hash::make($data['password']);
        }

        $user->save();

        return redirect()->route('settings.users.index')->with('success', 'تم تحديث المستخدم بنجاح.');
    }

    public function toggleActive($id)
    {

        $current = Auth::user();
        $user = User::where('clinic_id', $current->clinic_id)->findOrFail($id);

        $user->is_active = !$user->is_active;
        $user->save();

        return back()->with('success', 'تم تغيير حالة المستخدم.');
    }

    public function destroy($id)
    {
        $current = Auth::user();
        $user = User::where('clinic_id', $current->clinic_id)->findOrFail($id);

        $user->delete();

        return back()->with('success', 'تم حذف المستخدم.');
    }
}
