<?php

namespace App\Services;

use App\Models\ApiCallLog;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Str;

class RbMockService
{
    protected string $baseUrl;

    protected array $bodies;

    protected array $initHeaders;

    protected array $opCfg;

    public function __construct()
    {
        $this->baseUrl = rtrim(config('rb_mock.base_url'), '/');
        if (! $this->baseUrl) {
            throw new \RuntimeException('MOCK_BASE_URL is not set');
        }

        $this->bodies = config('rb_mock.bodies', []);
        $this->initHeaders = config('rb_mock.initial_offer_headers', []);
        $this->opCfg = config('rb_mock.operator_txn', [
            'send_in' => 'body',
            'header_name' => 'Operator-Transaction-Number',
            'body_key' => 'operatorTransactionNumber',
            'prefix' => 'RB',
        ]);
    }

    protected function generateOperatorTxn(): string
    {
        $id = (string) Str::ulid();
        $prefix = $this->opCfg['prefix'] ?? null;

        return $prefix ? "{$prefix}-{$id}" : $id;
    }

    protected function logAndReturn($req, string $method, string $path, ?array $body, array $finalHeaders, ?string $opTxn)
    {
        ApiCallLog::create([
            'provider' => 'rb-mock',
            'endpoint' => strtoupper($method).' '.$path,
            'http_method' => strtoupper($method),
            'http_code' => $req->status(),
            'request_headers' => $finalHeaders,
            'request_body' => $body,
            'response_headers' => $req->headers(),
            'response_body' => $req->body(),
            'success' => $req->successful(),
            'operator_txn' => $opTxn,
        ]);

        if (! $req->successful()) {
            throw new \RuntimeException("Mock error ({$req->status()}): ".$req->body());
        }

        $raw = $req->body();
        $json = json_decode($raw, true);

        return [
            'operatorTxn' => $opTxn,
            'data' => is_array($json) ? $json : ['raw' => $raw],

        ];
    }

    protected function attachOperatorTxn(?array $payload = null, ?string $opTxn = null, array $headers = []): array
    {
        $opTxn = $opTxn ?: $this->generateOperatorTxn();
        $sendIn = $this->opCfg['send_in'] ?? 'header';

        if (in_array($sendIn, ['header', 'both'], true)) {
            $headers[$this->opCfg['header_name'] ?? 'Operator-Transaction-Number'] = $opTxn;
        }
        if (in_array($sendIn, ['body', 'both'], true)) {
            $key = $this->opCfg['body_key'] ?? 'operatorTransactionNumber';
            $payload = is_array($payload) ? $payload : [];
            $payload[$key] = $opTxn;
        }

        return [$payload, $headers, $opTxn];
    }

    protected function post(string $path, array $body = [], array $headers = [], ?string $opTxn = null)
    {
        $url = $this->baseUrl.'/'.ltrim($path, '/');
        $h = array_merge(['Accept' => 'application/json', 'Content-Type' => 'application/json'], $headers);

        [$body, $h, $opTxn] = $this->attachOperatorTxn($body, $opTxn, $h);

        $req = Http::withHeaders($h)->post($url, $body);

        return $this->logAndReturn($req, 'POST', $path, $body, $h, $opTxn);
    }

    protected function get(string $path, array $headers = [], ?string $opTxn = null)
    {
        $url = $this->baseUrl.'/'.ltrim($path, '/');
        $h = array_merge(['Accept' => 'application/json'], $headers);

        [, $h, $opTxn] = $this->attachOperatorTxn(null, $opTxn, $h);

        $req = Http::withHeaders($h)->get($url);

        return $this->logAndReturn($req, 'GET', $path, null, $h, $opTxn);
    }

    public function offerRoughOffer(?array $body = null, ?string $opTxn = null): array
    {
        $payload = $body ?? $this->bodies['rough_offer'];

        return $this->post('offer/rough-offer', $payload, [], $opTxn);
    }

    public function offerInquiry(?array $body = null, ?string $opTxn = null): array
    {
        $payload = $body ?? $this->bodies['inquiry'];

        return $this->post('offer/inquiry', $payload, [], $opTxn);
    }

    public function offerExport(?array $body = null, ?string $opTxn = null): array
    {
        $payload = $body ?? $this->bodies['export'];

        return $this->post('offer/export', $payload, [], $opTxn);
    }

    public function offerInitialOffer(?array $body = null, ?string $opTxn = null): array
    {
        $payload = $body ?? $this->bodies['initial_offer'];
        $headers = array_merge($this->initHeaders, [
            'x-mock-match-request-body' => 'false',
        ]);

        return $this->post('offer/initial-offer', $payload, $headers, $opTxn);
    }

    public function offerDecideInitialOffer(string $offerId, ?array $body = null, ?string $opTxn = null): array
    {
        $payload = $body ?? $this->bodies['decide_initial_offer'];

        return $this->post("offer/initial-offer/{$offerId}/decision", $payload, [], $opTxn);
    }

    public function offerFinalOfferDecision(string $offerId, ?array $body = null, ?string $opTxn = null): array
    {
        $payload = $body ?? $this->bodies['final_offer_decision'];

        return $this->post("offer/final-offer/{$offerId}/decision", $payload, [], $opTxn);
    }

    public function offerCancel(string $offerId, ?array $body = null, ?string $opTxn = null): array
    {
        $payload = $body ?? $this->bodies['cancel_offer'];

        return $this->post("offer/cancel-offer/{$offerId}", $payload, [], $opTxn);
    }

    public function offerStatus(string $offerId, ?string $opTxn = null): array
    {
        return $this->get("offer/status/{$offerId}", [], $opTxn);
    }
}
